<?php
session_start();
require_once 'pdo_db.php';

function numberToWords($num) {
    $fmt = new NumberFormatter("en", NumberFormatter::SPELLOUT);
    return ucfirst($fmt->format($num)) . " only";
}

// Load available cheque leaves
$leaves = $pdo->query("SELECT id, leaf_number FROM cheque_leaves WHERE status = 'unused' ORDER BY leaf_number ASC")->fetchAll(PDO::FETCH_ASSOC);

// ✅ Abort early if no leaves exist
if (empty($leaves)) {
    die("<div style='padding:20px;color:red;font-weight:bold;'>❌ No unused cheque leaves available in the system.</div><a href='load_cheque_book.php'>➕ Load Cheque Book</a>");
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $leaf_id = (int)$_POST['cheque_leaf_id'];
    $payee = trim($_POST['payee_name']);
    $amount = floatval($_POST['amount']);
    $amount_words = numberToWords($amount);
    $date = $_POST['payment_date'];
    $reason = trim($_POST['reason']);
    $user_id = 1; // Replace with $_SESSION['user_id'] if using login

    // ✅ Confirm selected leaf ID is valid and unused
    $checkStmt = $pdo->prepare("SELECT id FROM cheque_leaves WHERE id = ? AND status = 'unused'");
    $checkStmt->execute([$leaf_id]);
    $leafExists = $checkStmt->fetchColumn();

    if (!$leafExists) {
        die("<div style='padding:20px;color:red;font-weight:bold;'>❌ Invalid or already used cheque leaf ID: $leaf_id</div><a href='add_cheque.php'>⬅ Try again</a>");
    }

    // Save cheque
    $stmt = $pdo->prepare("INSERT INTO cheques (cheque_leaf_id, payee_name, amount, amount_words, payment_date, reason, printed_by, printed_at, status)
                           VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), 'printed')");
    $stmt->execute([$leaf_id, $payee, $amount, $amount_words, $date, $reason, $user_id]);

    // Mark leaf as printed
    $pdo->prepare("UPDATE cheque_leaves SET status = 'printed' WHERE id = ?")->execute([$leaf_id]);

    $cheque_id = $pdo->lastInsertId();
    header("Location: print_cheque.php?cheque_id=$cheque_id");
    exit;
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>New Cheque</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <script>
    function updateWords() {
        const amount = parseFloat(document.getElementById('amount').value);
        if (!isNaN(amount)) {
            fetch('to_words.php?amount=' + amount)
                .then(res => res.text())
                .then(data => document.getElementById('amount_words').innerText = data);
        } else {
            document.getElementById('amount_words').innerText = '';
        }
    }
    </script>
</head>
<body class="container py-4">
    <h3>Create New Cheque</h3>

    <form method="POST">
        <div class="mb-3">
            <label class="form-label">Payee Name</label>
            <input type="text" name="payee_name" class="form-control" required>
        </div>

        <div class="mb-3">
            <label class="form-label">Amount</label>
            <input type="number" step="0.01" name="amount" id="amount" class="form-control" oninput="updateWords()" required>
            <small class="text-muted">In Words: <strong id="amount_words"></strong></small>
        </div>

        <div class="mb-3">
            <label class="form-label">Payment Date</label>
            <input type="date" name="payment_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
        </div>

        <div class="mb-3">
            <label class="form-label">Purpose / Memo</label>
            <textarea name="reason" class="form-control" rows="2"></textarea>
        </div>

        <div class="mb-3">
            <label class="form-label">Cheque Leaf</label>
            <select name="cheque_leaf_id" class="form-select" required>
                <?php foreach ($leaves as $i => $leaf): ?>
                    <option value="<?= $leaf['id'] ?>" <?= $i === 0 ? 'selected' : '' ?>>
                        <?= $i === 0 ? '[Auto-Select] ' : 'Manual: ' ?>Leaf #<?= $leaf['leaf_number'] ?> (ID: <?= $leaf['id'] ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <button type="submit" class="btn btn-success">🖨 Print Cheque</button>
    </form>
</body>
</html>
