
<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';


// Fetch suppliers
$sql = "SELECT id, name FROM suppliers ORDER BY name";
$result = $conn->query($sql);
if (!$result) {
    die("Error fetching suppliers: " . $conn->error);
}
$suppliers = $result->fetch_all(MYSQLI_ASSOC);

// Fetch banks
$sql = "SELECT id, bank_name FROM banks ORDER BY bank_name";
$result = $conn->query($sql);
if (!$result) {
    die("Error fetching banks: " . $conn->error);
}
$banks = $result->fetch_all(MYSQLI_ASSOC);

// Fetch payments for reference
$sql = "SELECT id, reference_no FROM payments WHERE method = 'Cheque' AND cheque_id IS NULL ORDER BY created_at DESC";
$result = $conn->query($sql);
if (!$result) {
    die("Error fetching payments: " . $conn->error);
}
$payments = $result->fetch_all(MYSQLI_ASSOC);

// Amount to words function
function numberToWords($number) {
    $ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine'];
    $teens = ['Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'];
    $tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
    $thousands = ['', 'Thousand', 'Million', 'Billion'];

    if ($number == 0) return 'Zero';

    $words = '';
    $number = (float)$number;
    $integer = floor($number);
    $decimal = round(($number - $integer) * 100);

    $chunk = 0;
    while ($integer > 0) {
        $part = $integer % 1000;
        $integer = floor($integer / 1000);
        if ($part > 0) {
            $partWords = '';
            if ($part >= 100) {
                $partWords .= $ones[floor($part / 100)] . ' Hundred ';
                $part %= 100;
            }
            if ($part >= 20) {
                $partWords .= $tens[floor($part / 10)] . ' ';
                $part %= 10;
            }
            if ($part >= 10 && $part < 20) {
                $partWords .= $teens[$part - 10] . ' ';
                $part = 0;
            }
            if ($part > 0) {
                $partWords .= $ones[$part] . ' ';
            }
            $words = $partWords . $thousands[$chunk] . ' ' . $words;
        }
        $chunk++;
    }

    if ($decimal > 0) {
        $words .= 'and ';
        if ($decimal >= 20) {
            $words .= $tens[floor($decimal / 10)] . ' ';
            $decimal %= 10;
        }
        if ($decimal >= 10 && $decimal < 20) {
            $words .= $teens[$decimal - 10] . ' ';
            $decimal = 0;
        }
        if ($decimal > 0) {
            $words .= $ones[$decimal] . ' ';
        }
        $words .= 'Cents';
    }

    return trim($words);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $cheque_no = $_POST['cheque_no'] ?? '';
    $supplier_id = $_POST['supplier_id'] ?? null;
    $payee = $_POST['payee_manual'] ?? ($_POST['payee'] ?? '');
    $bank_id = $_POST['bank_id'] ?? null;
    $amount = $_POST['amount'] ?? 0;
    $cheque_date = $_POST['cheque_date'] ?? date('Y-m-d');
    $remarks = $_POST['remarks'] ?? '';
    $ac_payee = isset($_POST['ac_payee']) ? 1 : 0;
    $payment_id = $_POST['payment_id'] ?? null;
    $amount_words = numberToWords($amount);

    // Validate inputs
    if (empty($cheque_no) || empty($payee) || empty($bank_id) || $amount <= 0) {
        $error = "Cheque number, payee, bank, and amount are required.";
    } else {
        // Save to database
        $sql = "INSERT INTO cheques (cheque_no, payee, supplier_id, bank_id, amount, amount_words, cheque_date, remarks, ac_payee, status, payment_id, created_by, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'Issued', ?, ?, NOW())";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param('ssiidsisiii', $cheque_no, $payee, $supplier_id, $bank_id, $amount, $amount_words, $cheque_date, $remarks, $ac_payee, $payment_id, $_SESSION['user_id']);
        if ($stmt->execute()) {
            $cheque_id = $conn->insert_id;
            if ($payment_id) {
                // Update payments with cheque_id
                $sql = "UPDATE payments SET cheque_id = ? WHERE id = ? AND created_by = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param('iii', $cheque_id, $payment_id, $_SESSION['user_id']);
                $stmt->execute();
            }
            header('Location: print.php?id=' . $cheque_id);
            exit;
        } else {
            $error = "Failed to save cheque: " . $conn->error;
        }
    }
}
?>


<div class="container-fluid">
    <h2 class="my-4"><i class="fas fa-pen me-2"></i> Write Cheque</h2>
    <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    <form method="POST" class="card p-4 shadow-sm">
        <div class="mb-3">
            <label for="cheque_no" class="form-label">Cheque Number</label>
            <input type="text" class="form-control" id="cheque_no" name="cheque_no" required>
        </div>
        <div class="mb-3">
            <label for="payee" class="form-label">Payee</label>
            <select class="form-select" id="payee" name="supplier_id" onchange="togglePayeeInput()">
                <option value="">Select Supplier</option>
                <?php foreach ($suppliers as $supplier): ?>
                    <option value="<?php echo $supplier['id']; ?>" data-name="<?php echo htmlspecialchars($supplier['name']); ?>">
                        <?php echo htmlspecialchars($supplier['name']); ?>
                    </option>
                <?php endforeach; ?>
                <option value="manual">Manual Entry</option>
            </select>
            <input type="text" class="form-control mt-2 d-none" id="payee_manual" name="payee_manual" placeholder="Enter Payee Name">
        </div>
        <div class="mb-3">
            <label for="bank_id" class="form-label">Bank</label>
            <select class="form-select" id="bank_id" name="bank_id" required>
                <option value="">Select Bank</option>
                <?php foreach ($banks as $bank): ?>
                    <option value="<?php echo $bank['id']; ?>">
                        <?php echo htmlspecialchars($bank['bank_name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="payment_id" class="form-label">Link to Payment (Optional)</label>
            <select class="form-select" id="payment_id" name="payment_id" onchange="updateRemarks()">
                <option value="">Select Payment</option>
                <?php foreach ($payments as $payment): ?>
                    <option value="<?php echo $payment['id']; ?>" data-reference="<?php echo htmlspecialchars($payment['reference_no']); ?>">
                        <?php echo htmlspecialchars($payment['reference_no']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="amount" class="form-label">Amount</label>
            <input type="number" step="0.01" class="form-control" id="amount" name="amount" required>
            <small id="amount_words" class="form-text text-muted">Enter amount to see words</small>
        </div>
        <div class="mb-3">
            <label for="cheque_date" class="form-label">Cheque Date</label>
            <input type="date" class="form-control" id="cheque_date" name="cheque_date" value="<?php echo date('Y-m-d'); ?>" required>
        </div>
        <div class="mb-3">
            <label for="remarks" class="form-label">Remarks (Optional)</label>
            <input type="text" class="form-control" id="remarks" name="remarks" placeholder="e.g., Payment for REF123">
        </div>
        <div class="mb-3 form-check">
            <input type="checkbox" class="form-check-input" id="ac_payee" name="ac_payee">
            <label class="form-check-label" for="ac_payee">A/C Payee Only</label>
        </div>
        <button type="submit" class="btn btn-primary"><i class="fas fa-save me-2"></i> Save & Print</button>
    </form>
</div>

<script>
// Toggle payee manual input
function togglePayeeInput() {
    const select = document.getElementById('payee');
    const manualInput = document.getElementById('payee_manual');
    if (select.value === 'manual') {
        manualInput.classList.remove('d-none');
        manualInput.required = true;
    } else {
        manualInput.classList.add('d-none');
        manualInput.required = false;
        manualInput.value = select.options[select.selectedIndex].dataset.name || '';
    }
}

// Update remarks with payment reference
function updateRemarks() {
    const paymentSelect = document.getElementById('payment_id');
    const remarksInput = document.getElementById('remarks');
    if (paymentSelect.value) {
        const reference = paymentSelect.options[paymentSelect.selectedIndex].dataset.reference;
        remarksInput.value = `Payment for ${reference}`;
    } else {
        remarksInput.value = '';
    }
}

// Real-time amount to words preview
document.getElementById('amount').addEventListener('input', function() {
    const amount = parseFloat(this.value) || 0;
    fetch('/smartpay360/modules/cheques/convert_amount.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'amount=' + amount
    })
    .then(response => response.text())
    .then(words => {
        document.getElementById('amount_words').innerText = words || 'Enter amount';
    });
});
</script>

<?php include '../../includes/footer.php'; ?>