<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

if (!isset($_GET['id'])) {
    $_SESSION['error'] = "Invoice not specified.";
    header("Location: list.php");
    exit;
}

$invoice_id = (int) $_GET['id'];

// Fetch invoice
$stmt = $conn->prepare("
    SELECT i.*, s.name AS supplier_name, b.branch_name
    FROM invoices i
    JOIN suppliers s ON i.supplier_id = s.id
    JOIN branches b ON i.branch_id = b.id
    WHERE i.id = ?
    LIMIT 1
");
$stmt->bind_param("i", $invoice_id);
$stmt->execute();
$result = $stmt->get_result();
if ($result->num_rows === 0) {
    $_SESSION['error'] = "Invoice not found.";
    header("Location: list.php");
    exit;
}
$row = $result->fetch_assoc();

// Payment allocations
$payment_stmt = $conn->prepare("
    SELECT p.id, p.reference_no, p.payment_date, p.method, p.remarks, pa.amount_allocated
    FROM payment_allocations pa
    JOIN payments p ON pa.payment_id = p.id
    WHERE pa.invoice_id = ?
    ORDER BY p.payment_date DESC
");
$payment_stmt->bind_param("i", $invoice_id);
$payment_stmt->execute();
$payment_result = $payment_stmt->get_result();
$payments = [];
$total_allocated = 0;
while ($p = $payment_result->fetch_assoc()) {
    $total_allocated += $p['amount_allocated'];
    $payments[] = $p;
}

// Credit notes
$credit_stmt = $conn->prepare("SELECT id, amount, created_at FROM credit_notes WHERE invoice_id = ?");
$credit_stmt->bind_param("i", $invoice_id);
$credit_stmt->execute();
$credit_result = $credit_stmt->get_result();
$credits = [];
$total_credit = 0;
while ($c = $credit_result->fetch_assoc()) {
    $total_credit += $c['amount'];
    $credits[] = $c;
}

// Invoice logs
$log_stmt = $conn->prepare("
    SELECT l.*, u.full_name 
    FROM invoice_logs l
    JOIN users u ON l.user_id = u.id
    WHERE l.invoice_id = ?
    ORDER BY l.timestamp ASC
");
$log_stmt->bind_param("i", $invoice_id);
$log_stmt->execute();
$log_result = $log_stmt->get_result();
$logs = [];
while ($log = $log_result->fetch_assoc()) {
    $logs[] = $log;
}

$remaining_balance = $row['total_payable'] - $total_allocated - $total_credit;
?>

<div class="main-content">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h4 class="fw-bold text-primary"><i class="fas fa-eye me-2"></i> View Invoice #<?= htmlspecialchars($row['invoice_number']) ?></h4>
        <a href="list.php" class="btn btn-sm btn-secondary"><i class="fas fa-arrow-left me-1"></i>Back</a>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show"><?= $_SESSION['error']; unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <div class="card shadow-sm p-4 bg-white mb-4">
        <div class="row">
            <div class="col-md-6 mb-3"><h6>Supplier:</h6><p><?= htmlspecialchars($row['supplier_name']) ?></p></div>
            <div class="col-md-6 mb-3"><h6>Branch:</h6><p><?= htmlspecialchars($row['branch_name']) ?></p></div>
            <div class="col-md-4 mb-3"><h6>Invoice Date:</h6><p><?= date('d M Y', strtotime($row['invoice_date'])) ?></p></div>
            <div class="col-md-4 mb-3"><h6>Total Invoice:</h6><p>KSh <?= number_format($row['total_amount'], 2) ?></p></div>
            <div class="col-md-4 mb-3">
                <h6>Status:</h6>
                <?php
                $badgeClass = match ($row['status']) {
                    'Draft' => 'bg-secondary',
                    'Submitted' => 'bg-info text-dark',
                    'Verified' => 'bg-warning text-dark',
                    'Approved' => 'bg-success',
                    'Rejected' => 'bg-danger',
                    'Cancelled' => 'bg-dark',
                    default => 'bg-light text-dark'
                };
                ?>
                <span class="badge <?= $badgeClass ?>"><?= $row['status'] ?></span>
            </div>
        </div>

        <hr>

        <div class="row">
            <div class="col-md-3 mb-3"><strong>VATable Amount:</strong><br>KSh <?= number_format($row['vatable_amount'], 2) ?></div>
            <div class="col-md-3 mb-3"><strong>VAT (16%):</strong><br>KSh <?= number_format($row['vat_amount'], 2) ?></div>
            <div class="col-md-3 mb-3"><strong>Withholding Tax (2%):</strong><br>KSh <?= number_format($row['withholding_tax'], 2) ?></div>
            <div class="col-md-3 mb-3"><strong>Total Payable:</strong><br><span class="text-success fw-bold">KSh <?= number_format($row['total_payable'], 2) ?></span></div>
        </div>

        <div class="mt-3">
            <h6>Invoice File:</h6>
            <?php if (!empty($row['file_path'])): ?>
                <a class="btn btn-outline-primary btn-sm" href="../../assets/uploads/invoices/<?= $row['file_path'] ?>" target="_blank">
                    <i class="fas fa-file-pdf me-1"></i> View PDF
                </a>
            <?php else: ?>
                <span class="text-muted">No file uploaded.</span>
            <?php endif; ?>
        </div>

        <!-- Workflow Actions -->
        <div class="mt-4">
            <?php if ($row['status'] === 'Draft'): ?>
                <a href="transition_invoice.php?action=submit&id=<?= $row['id'] ?>" class="btn btn-warning"><i class="fas fa-paper-plane"></i> Submit for Verification</a>
                <a href="transition_invoice.php?action=cancel&id=<?= $row['id'] ?>" class="btn btn-outline-dark ms-2"><i class="fas fa-times-circle"></i> Cancel</a>
            <?php elseif ($row['status'] === 'Submitted'): ?>
                <a href="transition_invoice.php?action=verify&id=<?= $row['id'] ?>" class="btn btn-success"><i class="fas fa-check-double"></i> Verify</a>
                <a href="transition_invoice.php?action=reject&id=<?= $row['id'] ?>" class="btn btn-outline-danger ms-2"><i class="fas fa-ban"></i> Reject</a>
            <?php elseif ($row['status'] === 'Verified'): ?>
                <a href="transition_invoice.php?action=approve&id=<?= $row['id'] ?>" class="btn btn-primary"><i class="fas fa-thumbs-up"></i> Approve</a>
            <?php endif; ?>
        </div>
    </div>

    <!-- Payment Allocations -->
    <h5 class="text-primary mb-3"><i class="fas fa-money-check-alt me-2"></i>Payment Allocations (<?= count($payments) ?>)</h5>
    <div class="card shadow-sm border-0">
        <div class="card-body table-responsive">
            <table class="table table-hover table-bordered align-middle small">
                <thead class="table-dark">
                    <tr class="text-center">
                        <th>#</th>
                        <th>Payment ID</th>
                        <th>Reference No</th>
                        <th>Payment Date</th>
                        <th>Method</th>
                        <th>Allocated</th>
                        <th>Remarks</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($payments): foreach ($payments as $i => $p): ?>
                        <tr>
                            <td class="text-center"><?= $i + 1 ?></td>
                            <td class="text-center">
                                <a href="../payments/view.php?id=<?= $p['id'] ?>" class="text-primary"><?= $p['id'] ?></a>
                            </td>
                            <td><?= $p['reference_no'] ?></td>
                            <td class="text-center"><?= date('d-M-Y', strtotime($p['payment_date'])) ?></td>
                            <td><?= $p['method'] ?></td>
                            <td class="text-end text-success">KSh <?= number_format($p['amount_allocated'], 2) ?></td>
                            <td><?= $p['remarks'] ?: 'N/A' ?></td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="7" class="text-center text-muted">No payments allocated.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Credit Notes -->
    <h5 class="text-primary mt-5 mb-3"><i class="fas fa-credit-card me-2"></i>Credit Notes (<?= count($credits) ?>)</h5>
    <div class="card shadow-sm border-0">
        <div class="card-body table-responsive">
            <table class="table table-hover table-bordered align-middle small">
                <thead class="table-dark">
                    <tr class="text-center">
                        <th>#</th>
                        <th>Credit Note ID</th>
                        <th>Amount</th>
                        <th>Created At</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($credits): foreach ($credits as $i => $c): ?>
                        <tr>
                            <td class="text-center"><?= $i + 1 ?></td>
                            <td class="text-center"><?= $c['id'] ?></td>
                            <td class="text-end text-info">KSh <?= number_format($c['amount'], 2) ?></td>
                            <td class="text-center"><?= date('d-M-Y', strtotime($c['created_at'])) ?></td>
                        </tr>
                    <?php endforeach; else: ?>
                        <tr><td colspan="4" class="text-center text-muted">No credit notes.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
            <div class="mt-3 text-end">
                <strong>Total Allocated:</strong> KSh <?= number_format($total_allocated, 2) ?><br>
                <strong>Total Credit:</strong> KSh <?= number_format($total_credit, 2) ?><br>
                <strong>Remaining Balance:</strong> <span class="text-<?= $remaining_balance <= 0 ? 'success' : 'warning' ?> fw-bold">KSh <?= number_format($remaining_balance, 2) ?></span>
            </div>
        </div>
    </div>

   <!-- Workflow History -->
<h5 class="text-primary mt-5 mb-3"><i class="fas fa-history me-2"></i>Workflow History</h5>
<div class="card shadow-sm border-0">
    <div class="card-body table-responsive">
        <table class="table table-bordered table-sm small">
            <thead class="table-dark">
                <tr class="text-center">
                    <th>#</th>
                    <th>Action</th>
                    <th>Performed By</th>
                    <th>Remarks</th>
                    <th>Date & Time</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($logs): foreach ($logs as $i => $log): ?>
                    <tr>
                        <td class="text-center"><?= $i + 1 ?></td>
                        <td class="text-center"><?= htmlspecialchars($log['action']) ?></td>
                        <td><?= htmlspecialchars($log['full_name']) ?></td>
                        <td><?= htmlspecialchars($log['remarks'] ?: '-') ?></td>
                        <td class="text-center"><?= date('d M Y, h:i A', strtotime($log['timestamp'])) ?></td>
                    </tr>
                <?php endforeach; else: ?>
                    <tr><td colspan="5" class="text-center text-muted">No log entries available.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>
<?php include '../../includes/footer.php'; ?>
