<?php
session_start();
require_once '../../config/db.php';

$suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name");

$filter_supplier = $_GET['supplier_id'] ?? '';
$filter_date = $_GET['date'] ?? date('Y-m-d');

$where = "DATE(l.log_time) = ?";
$params = [$filter_date];
$types = "s";

if ($filter_supplier) {
    $where .= " AND m.supplier_id = ?";
    $params[] = $filter_supplier;
    $types .= "i";
}

$stmt = $conn->prepare("
    SELECT 
        l.*, 
        m.full_name, m.id_number, m.id AS merchandiser_id, 
        s.name AS supplier_name, 
        b.branch_name,
        (SELECT log_type FROM merchandiser_logs WHERE merchandiser_id = m.id ORDER BY log_time DESC LIMIT 1) AS latest_log_type
    FROM merchandiser_logs l 
    JOIN merchandisers m ON l.merchandiser_id = m.id 
    JOIN suppliers s ON m.supplier_id = s.id 
    JOIN branches b ON l.branch_id = b.id 
    WHERE $where 
    ORDER BY l.log_time DESC
");
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();

include '../../includes/layout2.php';
?>

<div class="main-content">
  <h4 class="fw-bold mb-4 text-primary"><i class="fas fa-clock"></i> Merchandiser Log History</h4>

  <form method="GET" class="row g-3 mb-3">
    <div class="col-md-4">
      <label class="form-label">Filter by Supplier</label>
      <select name="supplier_id" class="form-select">
        <option value="">-- All Suppliers --</option>
        <?php while ($s = $suppliers->fetch_assoc()): ?>
          <option value="<?= $s['id'] ?>" <?= $filter_supplier == $s['id'] ? 'selected' : '' ?>>
            <?= htmlspecialchars($s['name']) ?>
          </option>
        <?php endwhile; ?>
      </select>
    </div>
    <div class="col-md-3">
      <label class="form-label">Filter by Date</label>
      <input type="date" name="date" value="<?= $filter_date ?>" class="form-control">
    </div>
    <div class="col-md-2 d-flex align-items-end">
      <button type="submit" class="btn btn-primary"><i class="fas fa-filter me-1"></i> Apply Filters</button>
    </div>
  </form>

  <div class="card p-3 shadow-sm">
    <div class="table-responsive">
      <table class="table table-bordered align-middle">
        <thead class="table-dark">
          <tr>
            <th>#</th>
            <th>Name</th>
            <th>ID</th>
            <th>Supplier</th>
            <th>Branch</th>
            <th>Type</th>
            <th>Time</th>
            <th>In-Store Time</th>
            <th>Action</th>
          </tr>
        </thead>
        <tbody>
          <?php $i = 1; while ($log = $result->fetch_assoc()): 
              $log_id = $log['id'];
              $log_time = strtotime($log['log_time']);
          ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= htmlspecialchars($log['full_name']) ?></td>
              <td><?= htmlspecialchars($log['id_number']) ?></td>
              <td><?= htmlspecialchars($log['supplier_name']) ?></td>
              <td><?= htmlspecialchars($log['branch_name']) ?></td>
              <td>
                <span class="badge bg-<?= $log['log_type'] === 'IN' ? 'success' : 'danger' ?>">
                  <?= $log['log_type'] ?>
                </span>
              </td>
              <td><?= date('d M Y H:i:s', $log_time) ?></td>
              <td>
                <?php if ($log['log_type'] === 'IN'): ?>
                  <span class="in-store-timer" data-logtime="<?= $log_time ?>"></span>
                <?php else: ?>
                  -
                <?php endif; ?>
              </td>
              <td>
                <button class="btn btn-sm btn-primary mb-1" onclick="printBadge(<?= $log_id ?>)">
                  <i class="fas fa-print me-1"></i> Print Badge
                </button>
                <?php if ($log['latest_log_type'] === 'IN'): ?>
                  <button class="btn btn-sm btn-danger" onclick="forceLogout(<?= $log['merchandiser_id'] ?>, <?= $log['branch_id'] ?>)">
                    <i class="fas fa-sign-out-alt me-1"></i> Force Logout
                  </button>
                <?php endif; ?>
              </td>
            </tr>
          <?php endwhile; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<script>
function printBadge(logId) {
    window.open('print_badge.php?log_id=' + logId, '_blank', 'width=400,height=600');
}

function forceLogout(merchandiserId, branchId) {
    if (!confirm('Are you sure you want to force logout this merchandiser?')) return;

    fetch('force_logout.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ merchandiser_id: merchandiserId, branch_id: branchId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Force logout successful!');
            location.reload();
        } else {
            alert('Failed: ' + data.message);
        }
    })
    .catch(() => {
        alert('Error contacting server.');
    });
}

function updateInStoreTimers() {
    const now = Math.floor(Date.now() / 1000);
    document.querySelectorAll('.in-store-timer').forEach(function(span) {
        const logTime = parseInt(span.getAttribute('data-logtime'));
        let diff = now - logTime;
        const hours = Math.floor(diff / 3600);
        diff %= 3600;
        const minutes = Math.floor(diff / 60);
        const seconds = diff % 60;

        let display = '';
        if (hours > 0) display += hours + 'h ';
        if (minutes > 0 || hours > 0) display += minutes + 'm ';
        display += seconds + 's';

        span.textContent = display;
    });
}

setInterval(updateInStoreTimers, 1000);
window.onload = updateInStoreTimers;
</script>
