<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

$today = date('Y-m-d');
$supplier_id = isset($_GET['supplier_id']) ? (int)$_GET['supplier_id'] : 0;
$from_date = $_GET['from_date'] ?? '';
$to_date = $_GET['to_date'] ?? '';
$method = $_GET['method'] ?? '';
$invoice_status = $_GET['invoice_status'] ?? '';
$reference = $_GET['reference'] ?? '';  // New reference filter

$conditions = [];
$params = [];
$types = '';

if ($supplier_id > 0) {
    $conditions[] = "p.supplier_id = ?";
    $params[] = $supplier_id;
    $types .= 'i';
}
if (!empty($from_date)) {
    $conditions[] = "p.payment_date >= ?";
    $params[] = $from_date;
    $types .= 's';
}
if (!empty($to_date)) {
    $conditions[] = "p.payment_date <= ?";
    $params[] = $to_date;
    $types .= 's';
}
if (!empty($method)) {
    $conditions[] = "p.method = ?";
    $params[] = $method;
    $types .= 's';
}
if (!empty($invoice_status)) {
    $conditions[] = "i.status = ?";
    $params[] = $invoice_status;
    $types .= 's';
}
if (!empty($reference)) {
    $conditions[] = "p.reference_no LIKE ?";
    $params[] = "%$reference%";
    $types .= 's';
}

$where = count($conditions) ? 'WHERE ' . implode(' AND ', $conditions) : '';

$query = "
    SELECT p.id, p.supplier_id, s.name AS supplier_name, p.amount, p.payment_date, p.created_at,
           p.reference_no, p.method,
           (SELECT GROUP_CONCAT(i.invoice_number ORDER BY i.invoice_number SEPARATOR ', ') 
            FROM payment_allocations pa 
            JOIN invoices i ON pa.invoice_id = i.id 
            WHERE pa.payment_id = p.id) AS invoice_numbers
    FROM payments p
    JOIN suppliers s ON p.supplier_id = s.id
    LEFT JOIN payment_allocations pa ON pa.payment_id = p.id
    LEFT JOIN invoices i ON pa.invoice_id = i.id
    $where
    ORDER BY p.payment_date DESC, p.created_at DESC
";

$stmt = $conn->prepare($query);
if (count($params) > 0) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

$suppliers = $conn->query("SELECT id, name FROM suppliers ORDER BY name ASC");
$invoice_statuses = ['Paid', 'Unpaid', 'Partially Paid']; // Assuming these statuses exist

$total_query = "SELECT SUM(p.amount) AS total_paid FROM payments p $where";
$total_stmt = $conn->prepare($total_query);
if (count($params) > 0) {
    $total_stmt->bind_param($types, ...$params);
}
$total_stmt->execute();
$total_result = $total_stmt->get_result()->fetch_assoc();
$total_paid = $total_result['total_paid'] ?? 0;
?>

<div class="main-content">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold text-primary"><i class="fas fa-credit-card me-2"></i> Supplier Payments</h4>
        <div class="btn-group">
            <a href="add.php" class="btn btn-sm btn-primary"><i class="fas fa-plus-circle me-1"></i> Add Payment</a>
            <a href="export_csv.php?<?= http_build_query($_GET) ?>" class="btn btn-sm btn-outline-success"><i class="fas fa-file-csv"></i> CSV</a>
            <a href="export_pdf.php?<?= http_build_query($_GET) ?>" class="btn btn-sm btn-outline-danger"><i class="fas fa-file-pdf"></i> PDF</a>
            <a href="statements.php" class="btn btn-sm btn-outline-secondary"><i class="fas fa-user"></i> Supplier Summary</a>
            <a href="statements.php" class="btn btn-sm btn-outline-secondary"><i class="fas fa-calendar-alt"></i> Monthly Summary</a>
        </div>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success']; unset($_SESSION['success']); ?></div>
    <?php elseif (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error']; unset($_SESSION['error']); ?></div>
    <?php endif; ?>

    <!-- Filter Section -->
    <div class="card mb-3 shadow-sm">
        <div class="card-body">
            <form method="GET" class="row g-2">
                <div class="col-md-3">
                    <select name="supplier_id" class="form-select form-select-sm">
                        <option value="">-- All Suppliers --</option>
                        <?php while ($supplier = $suppliers->fetch_assoc()): ?>
                            <option value="<?= $supplier['id'] ?>" <?= ($supplier_id == $supplier['id']) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($supplier['name']) ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <input type="date" name="from_date" value="<?= htmlspecialchars($from_date) ?>" class="form-control form-control-sm">
                </div>
                <div class="col-md-2">
                    <input type="date" name="to_date" value="<?= htmlspecialchars($to_date) ?>" class="form-control form-control-sm">
                </div>
                <div class="col-md-2">
                    <select name="method" class="form-select form-select-sm">
                        <option value="">-- Payment Method --</option>
                        <option value="cash" <?= ($method == 'cash') ? 'selected' : '' ?>>Cash</option>
                        <option value="mpesa" <?= ($method == 'mpesa') ? 'selected' : '' ?>>MPESA</option>
                        <option value="cheque" <?= ($method == 'cheque') ? 'selected' : '' ?>>Cheque</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <select name="invoice_status" class="form-select form-select-sm">
                        <option value="">-- Invoice Status --</option>
                        <?php foreach ($invoice_statuses as $status): ?>
                            <option value="<?= $status ?>" <?= ($invoice_status == $status) ? 'selected' : '' ?>><?= $status ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <input type="text" name="reference" value="<?= htmlspecialchars($reference) ?>" class="form-control form-control-sm" placeholder="Search by Reference">
                </div>
                <div class="col-md-2">
                    <button class="btn btn-sm btn-primary w-100">🔎 Filter</button>
                </div>
                <div class="col-md-2">
                    <a href="list.php" class="btn btn-sm btn-secondary w-100">🔄 Reset</a>
                </div>
            </form>
        </div>
    </div>

    <div class="alert alert-info fw-bold">💰 Total Paid: <span class="text-success">KSh <?= number_format($total_paid, 2) ?></span></div>

    <!-- Data Table -->
    <div class="card shadow-sm">
        <div class="card-body table-responsive">
            <table class="table table-sm table-bordered table-hover align-middle">
                <thead class="table-dark text-center">
                    <tr>
                        <th>#</th>
                        <th>Supplier</th>
                        <th>Date</th>
                        <th>Reference</th>
                        <th>Method</th>
                        <th>Amount</th>
                        <th>Invoices</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody class="text-center">
                    <?php if ($result->num_rows > 0): $i = 1; $last_reference = ''; ?>
                        <?php while ($row = $result->fetch_assoc()): ?>
                            <?php 
                                $is_today = ($row['payment_date'] == $today); 
                                $show_reference = ($row['reference_no'] != $last_reference);
                                $last_reference = $row['reference_no'];
                            ?>
                            <tr class="<?= $is_today ? 'table-success' : '' ?>">
                                <td><?= $i++ ?></td>
                                <td class="text-start"><?= htmlspecialchars($row['supplier_name']) ?></td>
                                <td><?= date('d-M-Y', strtotime($row['payment_date'])) ?></td>
                                <td><?= $show_reference ? htmlspecialchars($row['reference_no']) : '-' ?></td>
                                <td>
                                    <?php
                                        $method = strtolower($row['method']);
                                        $icon = match($method) {
                                            'cheque' => 'fas fa-money-check',
                                            'mpesa' => 'fas fa-mobile-alt',
                                            'cash' => 'fas fa-coins',
                                            default => 'fas fa-question-circle'
                                        };
                                    ?>
                                    <span class="badge bg-light text-dark"><i class="<?= $icon ?> me-1"></i> <?= strtoupper($method) ?></span>
                                </td>
                                <td class="text-end text-success fw-bold">KSh <?= number_format($row['amount'], 2) ?></td>
                                <td class="text-start"><?= htmlspecialchars($row['invoice_numbers'] ?: '-') ?></td>
                                <td>
                                    <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#viewModal<?= $row['id'] ?>">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <a href="receipt.php?id=<?= $row['id'] ?>" class="btn btn-sm btn-outline-secondary">
                                        <i class="fas fa-print"></i>
                                    </a>
                                </td>
                            </tr>

                            <!-- Modal for Payment Details -->
                            <div class="modal fade" id="viewModal<?= $row['id'] ?>" tabindex="-1" aria-labelledby="modalLabel<?= $row['id'] ?>" aria-hidden="true">
                              <div class="modal-dialog modal-dialog-centered">
                                <div class="modal-content">
                                  <div class="modal-header bg-primary text-white">
                                    <h5 class="modal-title" id="modalLabel<?= $row['id'] ?>"><i class="fas fa-receipt me-2"></i>Payment Details</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                  </div>
                                  <div class="modal-body text-start">
                                    <p><strong>Supplier:</strong> <?= htmlspecialchars($row['supplier_name']) ?></p>
                                    <p><strong>Reference:</strong> <?= htmlspecialchars($row['reference_no']) ?></p>
                                    <p><strong>Payment Method:</strong> <?= strtoupper($row['method']) ?></p>
                                    <p><strong>Amount:</strong> <span class="text-success fw-bold">KSh <?= number_format($row['amount'], 2) ?></span></p>
                                    <p><strong>Date:</strong> <?= date('d M Y', strtotime($row['payment_date'])) ?></p>
                                    <p><strong>Invoices:</strong> <?= htmlspecialchars($row['invoice_numbers']) ?></p>
                                  </div>
                                  <div class="modal-footer">
                                    <a href="receipt.php?id=<?= $row['id'] ?>" class="btn btn-outline-secondary"><i class="fas fa-print me-1"></i>Print</a>
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                  </div>
                                </div>
                              </div>
                            </div>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr><td colspan="8" class="text-muted">No payments found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php include '../../includes/footer.php'; ?>
