<?php
session_start();
require_once '../../config/db.php';
include '../../includes/layout2.php';

ini_set('display_errors', 1);
error_reporting(E_ALL);

$from = $_GET['from'] ?? date('Y-m-d', strtotime('-30 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$branch_filter = $_GET['branch'] ?? '';

$where = "WHERE h.report_date BETWEEN ? AND ?";
$params = [$from, $to];
$types = 'ss';

if ($branch_filter) {
    $where .= " AND b.branch_name = ?";
    $params[] = $branch_filter;
    $types .= 's';
}

// Fetch depositable reports
$sql = "SELECT h.id AS report_id, h.report_date, b.branch_name,
        COALESCE(SUM(d.drop1 + d.drop2 + d.drop3 + d.note_1000 + d.note_500 + d.note_200 + d.note_100 + d.note_50 + d.coins), 0) AS expected_cash,
        COALESCE(SUM(dep.deposit_amount), 0) AS banked_cash,
        (SELECT COALESCE(SUM(c.amount), 0) FROM till_report_cheques c WHERE c.report_id = h.id AND c.status = 'Unbanked') AS total_unbanked_cheques
        FROM till_report_headers h
        JOIN branches b ON b.id = h.branch_id
        LEFT JOIN till_report_details d ON d.report_id = h.id
        LEFT JOIN bank_deposits dep ON dep.report_id = h.id AND dep.deposit_type = 'Cash'
        $where
        GROUP BY h.id, b.branch_name, h.report_date
        ORDER BY h.report_date DESC";

$stmt = $conn->prepare($sql);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

$reports = [];
while ($r = $res->fetch_assoc()) {
    $r['remaining_cash'] = $r['expected_cash'] - $r['banked_cash'];
    $r['remaining_cheques'] = $r['total_unbanked_cheques'];
    $reports[] = $r;
}

$accounts = $conn->query("SELECT id, bank_name, account_number FROM bank_accounts WHERE status = 'active' ORDER BY bank_name");
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Record Bank Deposits</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body { background: #f8f9fa; font-size: 13px; }
    .header { background: #007bff; color: #fff; padding: 15px; margin-bottom: 20px; }
    .card { margin-bottom: 15px; }
    .cheque-list { max-height: 200px; overflow-y: auto; background: #fff; border: 1px solid #ccc; padding: 10px; }
  </style>
</head>
<body>
<div class="container my-3">
  <div class="header">
    <h5>🏦 Record Bank Deposits (Cash & Cheques)</h5>
  </div>

  <?php if (isset($_GET['success'])): ?>
    <div class="alert alert-success"><?= htmlspecialchars($_GET['success']) ?></div>
  <?php elseif (isset($_GET['error'])): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($_GET['error']) ?></div>
  <?php endif; ?>

  <!-- CASH DEPOSIT FORM -->
  <form action="save_cash_deposit.php" method="POST" enctype="multipart/form-data" class="card p-3 border-success">
    <h6 class="text-success">💵 Cash Deposit</h6>
    <div class="row mb-2">
      <div class="col-md-6">
        <label>Select Report</label>
        <select name="report_id" class="form-select" required>
          <option value="">-- Select Report --</option>
          <?php foreach ($reports as $r): if ($r['remaining_cash'] > 0): ?>
            <option value="<?= $r['report_id'] ?>">
              <?= $r['report_id'] ?> | <?= $r['branch_name'] ?> | <?= $r['report_date'] ?> (Cash Left: Ksh <?= number_format($r['remaining_cash'], 2) ?>)
            </option>
          <?php endif; endforeach; ?>
        </select>
      </div>
      <div class="col-md-3">
        <label>Amount (Ksh)</label>
        <input type="number" step="0.01" name="deposit_amount" class="form-control" required>
      </div>
      <div class="col-md-3">
        <label>Date</label>
        <input type="date" name="deposit_date" value="<?= date('Y-m-d') ?>" class="form-control" required>
      </div>
    </div>

    <div class="row mb-2">
      <div class="col-md-4">
        <label>Bank Account</label>
        <select name="account_id" class="form-select" required>
          <option value="">-- Select Account --</option>
          <?php $accounts->data_seek(0); while ($acc = $accounts->fetch_assoc()): ?>
            <option value="<?= $acc['id'] ?>"><?= $acc['bank_name'] ?> - <?= $acc['account_number'] ?></option>
          <?php endwhile; ?>
        </select>
      </div>
      <div class="col-md-4">
        <label>Reference No</label>
        <input type="text" name="reference_number" class="form-control" required>
      </div>
      <div class="col-md-4">
        <label>Deposited By</label>
        <input type="text" name="deposited_by" class="form-control" required>
      </div>
    </div>

    <div class="row mb-2">
      <div class="col-md-6">
        <label>Upload Deposit Slip (PDF)</label>
        <input type="file" name="deposit_slip" class="form-control" accept="application/pdf">
      </div>
      <div class="col-md-6">
        <label>Remarks</label>
        <input type="text" name="remarks" class="form-control">
      </div>
    </div>
    <input type="hidden" name="deposit_type" value="Cash">
    <button class="btn btn-success mt-2 float-end">💾 Save Cash Deposit</button>
  </form>

  <!-- CHEQUE DEPOSIT FORM -->
  <form action="save_cheque_deposit.php" method="POST" enctype="multipart/form-data" class="card p-3 border-primary">
    <h6 class="text-primary">🧾 Cheque Deposit</h6>
    <div class="row mb-2">
      <div class="col-md-12">
        <label>Select Report to Load Unbanked Cheques</label>
        <select id="report_cheque_selector" class="form-select">
          <option value="">-- Select Report --</option>
          <?php foreach ($reports as $r): if ($r['remaining_cheques'] > 0): ?>
            <option value="<?= $r['report_id'] ?>">
              <?= $r['report_id'] ?> | <?= $r['branch_name'] ?> | <?= $r['report_date'] ?> (Ksh <?= number_format($r['remaining_cheques'], 2) ?>)
            </option>
          <?php endif; endforeach; ?>
        </select>
      </div>
    </div>

    <div class="cheque-list mt-2" id="chequeContainer">
      <!-- Loaded via JS -->
    </div>

    <input type="hidden" name="report_id" id="selected_report_id">
    <input type="hidden" name="deposit_amount" id="cheque_total">
    <div class="row mt-2">
      <div class="col-md-3">
        <label>Total Amount</label>
        <input type="text" id="cheque_total_display" class="form-control" readonly>
      </div>
      <div class="col-md-3">
        <label>Date</label>
        <input type="date" name="deposit_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
      </div>
      <div class="col-md-3">
        <label>Reference No</label>
        <input type="text" name="reference_number" class="form-control" required>
      </div>
      <div class="col-md-3">
        <label>Deposited By</label>
        <input type="text" name="deposited_by" class="form-control" required>
      </div>
    </div>

    <div class="row mt-2">
      <div class="col-md-6">
        <label>Upload Deposit Slip</label>
        <input type="file" name="deposit_slip" class="form-control" accept="application/pdf">
      </div>
      <div class="col-md-6">
        <label>Bank Account</label>
        <select name="account_id" class="form-select" required>
          <option value="">-- Select Account --</option>
          <?php $accounts->data_seek(0); while ($acc = $accounts->fetch_assoc()): ?>
            <option value="<?= $acc['id'] ?>"><?= $acc['bank_name'] ?> - <?= $acc['account_number'] ?></option>
          <?php endwhile; ?>
        </select>
      </div>
    </div>

    <input type="hidden" name="deposit_type" value="Cheque">
    <button class="btn btn-primary mt-3 float-end">💾 Save Cheque Deposit</button>
  </form>

  <a href="deposits.php" class="btn btn-secondary mt-4">← Back to All Deposits</a>
</div>

<script>
document.getElementById('report_cheque_selector').addEventListener('change', function () {
  const reportId = this.value;
  document.getElementById('selected_report_id').value = reportId;

  fetch('fetch_cheques.php?report_id=' + reportId)
    .then(res => res.json())
    .then(data => {
      let html = '';
      let total = 0;
      data.forEach(cheque => {
        html += `<div class="form-check">
                   <input class="form-check-input cheque-item" type="checkbox" name="cheque_ids[]" value="${cheque.id}" data-amount="${cheque.amount}">
                   <label class="form-check-label">
                     ${cheque.customer_name} - Cheque ${cheque.cheque_number} - Ksh ${cheque.amount}
                   </label>
                 </div>`;
      });
      document.getElementById('chequeContainer').innerHTML = html;

      document.querySelectorAll('.cheque-item').forEach(box => {
        box.addEventListener('change', () => {
          let sum = 0;
          document.querySelectorAll('.cheque-item:checked').forEach(chk => {
            sum += parseFloat(chk.dataset.amount);
          });
          document.getElementById('cheque_total').value = sum.toFixed(2);
          document.getElementById('cheque_total_display').value = sum.toFixed(2);
        });
      });
    });
});
</script>
<?php if (isset($_GET['success'])): ?>
<!-- Success Modal -->
<div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content border-success">
      <div class="modal-header bg-success text-white">
        <h5 class="modal-title" id="successModalLabel"><i class="fas fa-check-circle me-2"></i>Success</h5>
        <button type="button" class="btn-close text-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        ✅ <?= htmlspecialchars($_GET['success']) ?>
      </div>
      <div class="modal-footer">
        <a href="record_deposit.php" class="btn btn-outline-success">OK</a>
      </div>
    </div>
  </div>
</div>

<script>
  const successModal = new bootstrap.Modal(document.getElementById('successModal'));
  successModal.show();
</script>
<?php endif; ?>

</body>
</html>
